module CM

import CMCombinators
import CMDatabase
import CMTypes
import CMUtilities
import CMVisualization
import StdEnv
import StdiTasks


// Main workflow
Start :: *World -> *World
Start world = workFlowTask [] public private world 
  where public :: Task ((Bool, UserId), User)
        public = initUsersDB #>> 
                 initPapersDB #>>
                 welcome
          
        private :: UserId User -> LabeledTask Void
        private _ user = ("Main", main user)
                                             
welcome :: Task ((Bool, UserId), User)
welcome = tryRepeat welcomeForm

welcomeForm :: Task (Maybe ((Bool, UserId), User))
welcomeForm = chooseTask msg [ ("Login", login)                          
                             , ("Submit a paper", submitPaper #>> return_V Nothing) 
                             ]
  where msg :: HtmlCode
        msg = [ H1 [] "Welcome", Br
              , Txt "Welcome to the iTask Conference Manager!", Br, Br
              , Txt "If you have been invited as a program committee member, you can "
              , Txt "log in to download papers. Everyone else is invited to submit a paper.", Br, Br
              ]
              
                 
// Public workflow
login :: Task (Maybe ((Bool, UserId), User))
login = newTask "login" (break (loginForm False))

loginForm :: Bool -> Task ((Bool, UserId), User)
loginForm fail = (msg ++ [Txt "Please enter your username and password:"])
                   ?>> editTask "Login" createDefault =>> \loginInfo ->
                 try (verifyUserDB loginInfo)
                     (\user -> updateUserDB user.uid_ (\user -> { user & new_ = False }) #>>
                               return_V ((user.new_, user.uid_), user)) 
                     (loginForm True)  
  where msg :: HtmlCode
        msg | fail      = [Txt "Login failed, please try again", Br, Br]
            | otherwise = []
                
submitPaper :: Task Void
submitPaper = msg ?>> breakable submitPaperForm
                                (\paper -> addPaperDB paper #>> 
                                           endMsg ?>> ok)
  where msg :: HtmlCode
        msg = [H1 [] "Please submit a paper:"]
        
        endMsg :: HtmlCode
        endMsg = [Txt "Thank you for submitting your paper"]                                                   

submitPaperForm :: Task Paper
submitPaperForm = msg ?>> editTaskPred {createDefault & title       = emptyTextInput
                                                      , author      = emptyTextInput
                                                      , affiliation = emptyTextInput
                                                      , email       = emptyTextInput
                                                      , url         = emptyTextInput
                                                      , abstract    = emptyTextArea} check <<@ Submit
  where msg :: HtmlCode
        msg = [Txt "Please fill in the following details about your submission:"] 
                     
        check :: Paper -> (Bool, HtmlCode)
        check paper | fromTextInput paper.title       == "" = (False, [Txt "You need to enter the title of your paper"])
                    | fromTextInput paper.author      == "" = (False, [Txt "You need to enter your name"])
                    | fromTextInput paper.Paper.email == "" = (False, [Txt "You need to enter your e-mail address"])
                    | fromTextInput paper.url         == "" = (False, [Txt "You need to enter the url where your paper can be found"])
                    | fromTextArea  paper.abstract    == "" = (False, [Txt "You need to enter the abstract of the paper"])
                    | otherwise                             = (True,[])             
                                                                                  

// Private workflow
main :: User -> Task Void
main user=:{User | login = {loginName}, role} = welcomeMsg ?>> foreverTask (chooseTask homeMsg userTasks)
  where welcomeMsg :: HtmlCode
        welcomeMsg  = [H1 [] ("Welcome " +++ loginName), Br]
   
        homeMsg :: HtmlCode
        homeMsg = [ Txt "Choose one of the tasks below or select a task that has been "
                  , Txt "assigned to you from the list on the left"
                  , Br, Br
                  ]
        
        userTasks :: [LabeledTask Void]
        userTasks = case role of
                      Chair -> [ ("Show users", showUsers)
                               , ("Add user", addUser)
                               , ("Show papers", showPapers)
                               , ("Assign reviewers", assignReviewers)
                               , ("Judge papers", judgePapers)
                               ]
                      PC    -> [ ("Show papers", showPapers)
                               , ("Mark papers", markPapers user.uid_)
                               ] 


// Chair workflow
addUser :: Task Void
addUser = breakable addUserForm
                    (\user -> addUserDB user #>> 
                              endMsg ?>> ok)
  where endMsg :: HtmlCode
        endMsg = [Txt "A new user has been added"]                               
               
addUserForm :: Task User
addUserForm = readUsersDB =>> \users ->
              msg ?>> editTaskPred {createDefault & User.email = emptyTextInput} 
                                   (check (map (\user -> user.User.login.loginName) users))
  where msg :: HtmlCode
        msg = [Txt "Please enter a username and password for the new user:"]    
  
        check :: [String] User -> (Bool, HtmlCode)
        check userNames user | user.User.login.loginName     == ""               = (False, [Txt "You need to enter a login name"])
                             | user.User.login.password      == (PasswordBox "") = (False, [Txt "You need to enter a password"])
                             | fromTextInput user.User.email == ""               = (False, [Txt "You need to enter an email address"])
                             | isMember user.User.login.loginName userNames      = (False, [Txt "This login name already exists"])
                             | otherwise                                         = (True,[])

showUsers :: Task Void
showUsers = readUsersDB =>> \users ->    
            formatUserList users ?>> ok
              
showPapers :: Task Void
showPapers = checkPapersDB $
             readPapersDB =>> \papers ->
             formatPaperList papers ?>> ok
    
assignReviewers :: Task Void
assignReviewers = checkUsersDB PC $
                  breakable (assignReviewersForm)
                            (\assignedReviewers -> endMsg ?>> addAssignedReviewersDB assignedReviewers)
  where endMsg :: HtmlCode
        endMsg = [Txt "Thank you for assigning reviewers"]                            
                           
assignReviewersForm :: Task [AssignedReviewer]
assignReviewersForm = readUsersByRoleDB PC =>> \users ->
                      readPapersDB =>> \papers ->
                      msg ?>> editTaskSubmit (toHtmlMatrix users papers) =>> \matrix ->
                      mapSt (\(paper, user) -> spawnWorkflow user.uid_ 
                                                             True 
                                                             ("Review " +++ toString paper.pid_, reviewPaper paper user.uid_))
                            (fromHtmlMatrix matrix) =>> \handles ->
                      return_V [  { paper_ = paper.pid_ 
                                  , user_  = user.uid_
                                  , handle = handle
                                  }
                               \\ (paper, user) <- fromHtmlMatrix matrix
                               &  handle <- handles
                               ]
  where msg :: HtmlCode
        msg = [Txt "Please assign reviewers:"]
                                   
        toHtmlMatrix :: [User] [Paper] -> HtmlMatrix (<|> HtmlPrompt (CheckBox, DisplayMode (Paper, User)))
        toHtmlMatrix users papers = HtmlMatrix paperTitles
                                               userNames 
                                               [  [ HtmlPrompt (mark paper user) <|> (CBNotChecked (printToString paper.pid_), HideMode (paper, user))
                                                  \\ user <- users 
                                                  ]
                                               \\ paper <- papers 
                                               ]
          where paperTitles :: [String]
                paperTitles = map (\x -> fromTextInput x.title) papers
                
                userNames :: [String]
                userNames = map (\x -> x.User.login.loginName) users
          
                mark :: Paper User -> String
                mark paper user = printToString (maybe id Neutral (lookup paper.pid_ user.marks_))
                                 
        fromHtmlMatrix :: (HtmlMatrix (<|> HtmlPrompt (CheckBox, DisplayMode (Paper, User)))) -> [(Paper, User)]
        fromHtmlMatrix (HtmlMatrix _ _ xss) = flatten [  [  (paper, user)
                                                         \\ (_ <|> (CBChecked _, HideMode (paper, user))) <- xs
                                                         ]
                                                      \\ xs <- xss
                                                      ]              
                       
judgePapers :: Task Void
judgePapers = readPapersDB =>> \papers ->
              splitAssignedReviewersDB papers =>> \papers=:(_, _, notJudged, _) ->
              case notJudged of
                []        -> (msg papers ++ [Txt "There are no papers that can be judged"]) ?>> ok
                notJudged -> breakable (chooseTask_pdm (msg papers ++ [Txt "Select a paper to judge:"]) 
                                                       -1 
                                                       (map (\paper -> (fromTextInput paper.title, return_V paper)) notJudged)
                                       )
                                       (\paper -> judgePaper paper #>> judgePapers)
  where msg :: ([Paper], [Paper], [Paper], [Paper]) -> HtmlCode
        msg (notAssigned, notReviewed, notJudged, judged) = notAssignedMsg notAssigned ++ 
                                                            notReviewedMsg notReviewed ++ 
                                                            notJudgedMsg notJudged ++ 
                                                            judgedMsg judged
  
        notAssignedMsg :: ([Paper] -> HtmlCode)
        notAssignedMsg = checkPapersLength [H4 [] "The following papers have not been assigned to reviewers yet:", Br]
                                
        notReviewedMsg :: ([Paper] -> HtmlCode)
        notReviewedMsg = checkPapersLength [H4 [] "The following papers have not been reviewed yet:", Br]
                                
        notJudgedMsg :: ([Paper] -> HtmlCode)
        notJudgedMsg = checkPapersLength [H4 [] "The following papers have not been judged yet:", Br]

        judgedMsg :: ([Paper] -> HtmlCode)
        judgedMsg = checkPapersLength [H4 [] "The following papers have already been judged:", Br]
                         
        checkPapersLength :: HtmlCode [Paper] -> HtmlCode
        checkPapersLength _      []     = []
        checkPapersLength prompt papers = prompt ++ formatPaperList papers ++ [Br]                                                     
                                     
judgePaper :: Paper -> Task Void
judgePaper paper = msg ?>> breakable (judgePaperForm paper)
                                     addJudgmentDB                                                  
                                     
  where msg :: HtmlCode
        msg = [Txt "Judge the following paper:"] ++ 
              toTable paperFormat paper ++
              [Br]
              
judgePaperForm :: Paper -> Task Judgment
judgePaperForm paper = lookupReviewsDB paper =>> \reviews ->  
                       msg reviews ?>> editTask "Ok" { createDefault & Judgment.paper_ = paper.pid_ }       
  where msg :: [Review] -> HtmlCode
        msg reviews = [Txt "This paper has received the following reviews:"] ++ 
                      formatReviewList reviews                                 


// PC Member workflow
markPapers :: UserId -> Task Void
markPapers uid_ = checkPapersDB $
                  breakable (markPapersForm uid_)
                            (updateUserConflictsDB uid_)

markPapersForm :: UserId -> Task [(PaperId, Mark)]
markPapersForm uid_ = lookupUserDB uid_ =>> \user ->
                      readPapersDB =>> \papers -> 
                      msg ?>> editTaskSubmit (VerList (toPromptPapers papers user.marks_)) =>> \(VerList marks_) ->
                      return_V (fromPromptPapers papers marks_)
  where msg :: HtmlCode
        msg = [ Txt "Please give each paper a mark:"
              , Ul [] [ Li [] [Txt "conflict of interest (Conflict)"]
                      , Li [] [Txt "no strong opinion (Neutral)"]
                      , Li [] [Txt "no knowledge of the domain (NoKnowledge)"]
                      , Li [] [Txt "strong preference (Preference)"]
                      ]
              , Hr [], Br
              ]
  
        toPromptPapers :: [Paper] [(PaperId, Mark)] -> [(HtmlPaper, Mark)]
        toPromptPapers []             _     = []
        toPromptPapers [paper:papers] marks = case lookup paper.pid_ marks of
                                                Nothing   -> [(HtmlPaper paper, Neutral) : toPromptPapers papers marks]
                                                Just mark -> [(HtmlPaper paper, mark)    : toPromptPapers papers marks]  
                                               
        fromPromptPapers :: [Paper] [(HtmlPaper, Mark)] -> [(PaperId, Mark)]
        fromPromptPapers papers marks = [ (paper.pid_, mark) \\ paper <- papers & (_, mark) <- marks ]                                   
                
reviewPaper :: Paper UserId -> Task Void
reviewPaper paper uid_ = reviewPaperForm paper =>> \review ->
                         addReviewDB paper.pid_ uid_ review #>> 
                         endMsg ?>> ok
  where endMsg :: HtmlCode
        endMsg = [Txt "Thank you for this review"]                         
        
reviewPaperForm :: Paper -> Task Review
reviewPaperForm paper = msg ?>> editTaskPred { createDefault & expertise      = toRadioGroup expertiseScale
                                                             , verdict        = toRadioGroup verdictScale
                                                             , commentsAuthor = emptyTextArea
                                                             , commentsPC     = emptyTextArea
                                             } check
  where msg :: HtmlCode
        msg = [Txt "You have been selected to review the following paper:", Br] ++
              toTable paperFormat paper ++
              [Br, Txt "You can download the paper ", A [Lnk_Href (fromTextInput paper.url)] [Txt "here"], Br, Br]
  
        check :: Review -> (Bool, HtmlCode)
        check review | fromTextArea review.commentsAuthor == "" = (False, [Txt "You need to enter comments for the author"])
                     | fromTextArea review.commentsPC     == "" = (False, [Txt "You need to enter comments for the PC members"])
                     | otherwise                                = (True, [])